/*global define */
/*
	Handle:

	This object implements the handle hierarchy.
*/


// TODO: create a mixin to remove repetition between Node and Handle object
define(["src/math/Mat3", "src/math/Vec2", "src/utils",
			"src/build/treeLeaf", "src/build/treeContainer", "lodash", "src/build/StageId",
			"lib/dev", "lib/dev/disableWhen"],
function (m3, v2, utils,
			treeLeaf, treeContainer, lodash, StageId,
			dev) {
	"use strict";

	// set to true to enable various sanity checks
	// set to false for faster performance wihout sanity checks
	var kVERIFY = dev.Debug.enabled,
		arrayPush = Array.prototype.push,
		genericSetParent,
		kOriginLeaf = "__Origin__Leaf__";

	function fnConstant (value) {
		return function () { return value; };
	}

	function rotationAngle (rotation) {
		var angle = Math.atan2(rotation[1], rotation[0]);
		return angle;
	}

	function verifyMatrix_(mat) {
		var i,
			n = mat.length;

		if (n !== 9) {
			throw new Error("verifyMatrix: local not 3x3 matrix.");
		}

		for (i = 0; i < n; i += 1) {
			if (typeof mat[i] !== "number") {
				throw new Error("verifyMatrix: non-number entry.");
			}

			if (Number.isNaN(mat[i])) {
				throw new Error("verifyMatrix: NaN entry.");
			}
		}
	}

	function decomposeAffine(mat, position, scale, shear, rotation) {
		m3.decomposeAffine(mat, position, scale, shear, rotation);

		// verify (if input matrix is singular recompose matrix will not match)
		if (kVERIFY && m3.isNonSingular(mat)) {
			var angle, recomposed;
			angle = rotationAngle(rotation);
			recomposed = m3.affine(position, scale, shear, angle, []);
			if (!m3.equalsApproximately(mat, recomposed)) {
				throw new Error("decomposeAffine: recompose matrix doesn't match input matrix");
			}
		}
	}

	function validateAutomation(handle) {
		utils.assert((lodash.has(handle.pAutomation, "touched") && lodash.size(handle.pAutomation) === 3) ||
						lodash.size(handle.pAutomation) === 2, "auto(): invalid key.");
	}


	function Handle(options0) {
		var options = lodash.defaults({}, options0, {
			name : "unnamed Handle",
			puppet : null,
			originParent_Local : [0, 0],
			origin : [0, 0], 
			locations : [ [0, 0] ],
			anchor : [0, 0],
			automate : { translation : true, linear : true},
			tagBackstageId : null
		});

		this.name_ = options.name;
		this.tagBackstageId_ = options.tagBackstageId;
		this.parentDagPath_ = options.parentDagPath;
		this.puppet_ = options.puppet;
		this.origin_ = options.origin;
		this.matParent_Local_ = m3.translation(options.originParent_Local);

		this.locations_ = options.locations;
		this.anchor_ = options.anchor;

		// represent transform as a local matrix, BUT
		// this may be inconvenient for keeping track of wind-up rotation which could be needed for motion blur or speedlines 
		this.local_ = m3.translation(options.origin);
		if (kVERIFY) { verifyMatrix_(this.local_); }

		this.pAutomation = options.automate;
		this.parent_ = null;
		this.children_ = [];

		this.StageId("Handle_");
	}

	Handle.kOriginLeaf = kOriginLeaf;

	utils.mixin(Handle, treeLeaf(), treeContainer(), StageId);
	genericSetParent = Handle.prototype.setParent;

	utils.mixin(Handle, {
		setName: function (name) {
			this.name_ = name;
		},

		// const return value
		getName: function () {
			return this.name_;
		},

		isOriginLeaf: function () {
			return this.getName() === kOriginLeaf;
		},

		getTagBackstageId : function () {
			var id = this.tagBackstageId_;
			return id ? id.slice(0) : null;
		},

		setParentDagPath : function (parentDagPath) {
			this.parentDagPath_ = parentDagPath;
		},

		getParentDagPath : function () {
			var path = this.parentDagPath_;
			return path ? path.slice(0) : null;
		},

		// Needed by Dragger 2.  Can be deleted after we decide to no longer
		// support dragger recordings created in Preview 3.
		getDagPathIdOld : function () {
			var p = this.getParentDagPath(),
				id = this.getTagBackstageId();
			if (p && id) {
				return p + "/" + id;
			}
			utils.assert(!p && id, "invalid dag path"); // if both not valid, we should have a valid backstage id. -jacquave
			return id;
		},

		getDagPathId : function () {
			var p = this.getParentDagPath(),
				id = this.getTagBackstageId();
			if (p && id) {
				return p + ":" + id;
			}
			utils.assert(!p && id, "invalid dag path"); // if both not valid, we should have a valid backstage id. -jacquave
			return ":" + id;
		},

		setPuppet: function (p) {
			this.puppet_ = p;
		},

		getPuppet: function () {
			return this.puppet_;
		},

		// HACK: refactor along with parentLayer and parentPuppet fields.
		getWarperLayer : function () {
			return this.getPuppet().getWarperLayer();
		},

		// TODO: remove as it no longer appears to need this override
		// overrides generic Tree default
		setParent: function (parent) {
			genericSetParent.call(this, parent);
		},

		getMatrixAtRestRelativeToParent : function (result0) {
			var result = m3.translation(this.origin_, result0);
			return m3.multiply(this.matParent_Local_, result, result);
		},

		/**
		 * Get origin leaf proxy if there is one.
		 */
		getOriginLeafProxy: function () {
			var h = this;
			
			if (h.isOriginLeaf()) {
				h = h.getParent() || h;
			}
			return h;
		},
		
		/**
		 * Set automation property for the handle.
		 * Destructively modifies current setting.  Consider auto() for use within behaviors.
		 * @param setting Desired values specified with two keys (translation and linear) and the corresponding value (true or false).
		 *
		 * Leaving either translation or linear unspecified leaves that setting unchanged.
		 */
		privateAuto: function (setting) {
			lodash.assign(this.pAutomation, setting);
			validateAutomation(this);
		},

		/**
		 * Return automation property as a logically ORed attribute.
		 */
		getAutoAttribute : function () {
			/*jshint bitwise: false */
			var bit = 0,
				attribute = 0;

			attribute |= this.pAutomation.translation << bit++;
			attribute |= this.pAutomation.linear << bit++;

			return attribute;
		},

		clone : function (clone_children, result) {
			if (result) {
				// init
				Handle.call(result);
			} else {
				// alloc and init
				result = new Handle();
			}

			result.name_ = this.name_.slice(0);
			result.matParent_Local_ = m3.clone(this.matParent_Local_);
			result.origin_ = v2.clone(this.origin_);
			result.local_ = m3.clone(this.local_);
			result.pAutomation = lodash.clone(this.pAutomation);
			result.tagBackstageId_ = this.getTagBackstageId();
			result.parentDagPath_ = this.getParentDagPath();

			result.locations_ = lodash.cloneDeep(this.locations_);
			result.anchor_ = lodash.clone(this.anchor_);

			// puppet must be set explicitly after construction
			result.puppet_ = null; 

			// parent must be set explicitly after construction
			delete result.parent_;	// remove the parent...
			result.setParent(null);	// ...so that this call works correctly
			
			// deep clone (by default) all descendent handles
			clone_children = typeof clone_children === "boolean" ? clone_children : true;
			if (clone_children) {
				result.children_ = lodash.map(this.children_, function (hi) {
					var handleChild = hi.clone(true);
					handleChild.setParent(result);
					return handleChild;
				});
			}

			return result;
		},

		gatherHandleTreeArray : function () {
			var aHandle = [],
				aParentRef = [];

			// Just like preOrderEach but modified to create array of parent indices
			// DRY (?): if needed, refactor into preOrderEach by providing additional
			// arguments to the callback: arg, argIndex, parentIndex, ...
			var arg = this,
				parentRef = null,
				stackArg = [],
				stackRef = [];

			while (arg) {
				aHandle.push(arg);
				parentRef = aParentRef.push(parentRef) - 1;

				var kids = arg.getChildren && arg.getChildren().slice(0);
				arg = kids.shift();
				if (arg) {
					arrayPush.apply(stackArg, kids.reverse());
					arrayPush.apply(stackRef, kids.map(fnConstant(parentRef)));
				} else {
					arg = stackArg.pop();
					parentRef = stackRef.pop();
				}
			}

			return {
				aHandle : aHandle,
				aParentRef : aParentRef
			};
		},

		transformLocations : function (mat) {
			var result = this.locations_.map(function (l) {
				return v2.transformAffine(mat, l);
			});
			return result;
		},

		transformAnchor : function (mat, result0) {
			return v2.transformAffine(mat, this.anchor_, result0);
		}
	});

	return Handle;

});
